<?php
/**
 * Plugin Name: WarStyles — Style Engine for WarForms
 * Description: Color theming (no layout changes) for WarForms Reboot. Per-form presets + safe color overrides, delivered as CSS variables. Ships as a separate add-on so core stays stable.
 * Version: 0.4.0
 * Author: Kevin R. Ray
 * Requires at least: 6.0
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) exit;

define('WARSTYLES_VERSION', '0.4.1');
define('WARSTYLES_FILE', __FILE__);
define('WARSTYLES_DIR', plugin_dir_path(__FILE__));
define('WARSTYLES_URL', plugin_dir_url(__FILE__));

/**
 * Basic dependency check: WarForms Reboot uses post type 'warform' and shortcode [warform].
 * We don't hard-fail; we just don't activate features if core isn't present.
 */
function warstyles_is_warforms_present(): bool {
    return post_type_exists('warform') && shortcode_exists('warform');
}


/**
 * Integrated Builder Tab (preferred).
 * When WarForms provides the tabbed builder UI, WarStyles registers its own tab and renders inside it.
 */
add_filter('warforms_admin_builder_tabs', function($tabs, $post_id) {
    if (!warstyles_is_warforms_present()) return $tabs;
    if (!is_array($tabs)) $tabs = [];
    $tabs['warstyles'] = [
        'label' => 'WarStyles',
        'priority' => 30,
    ];
    return $tabs;
}, 20, 2);

add_action('warforms_admin_builder_render_tab_warstyles', function($post_id) {
    if (!warstyles_is_warforms_present()) return;
    $post = get_post((int)$post_id);
    if (!$post || !($post instanceof WP_Post)) return;
    echo '<div class="warstyles-panel">';
    warstyles_render_metabox($post);
    echo '</div>';
}, 10, 1);

/**
 * Preset registry.
 * IMPORTANT: Color-only tokens. DO NOT include layout values (padding/margins/display).
 */
function warstyles_get_presets(): array {
    $presets = [
        // Brand-ish defaults (dark, neutral, crisp).
        'warchief' => [
            'label' => 'Warchief (default)',
            'vars'  => [
                '--wf-form-bg'       => 'rgba(2,6,23,.18)',
                '--wf-form-bg2'      => 'rgba(2,6,23,.30)',
                '--wf-form-border'   => 'rgba(148,163,184,.20)',
                '--wf-form-text'     => '#e5edf7',
                '--wf-label'         => '#dbe6ff',
                '--wf-muted'         => 'rgba(156,170,199,.78)',

                '--wf-field-bg'      => 'rgba(2,6,23,.22)',
                '--wf-field-text'    => '#e5edf7',
                '--wf-field-border'  => 'rgba(148,163,184,.24)',
                '--wf-placeholder'   => 'rgba(156,170,199,.72)',

                '--wf-accent'        => '#38bdf8',

                '--wf-button-bg'     => 'rgba(56,189,248,.18)',
                '--wf-button-text'   => '#e5edf7',
                '--wf-button-border' => 'rgba(56,189,248,.58)',

                '--wf-notice-bg'     => 'rgba(2,6,23,.26)',
                '--wf-notice-border' => 'rgba(148,163,184,.18)',
            ],
        ],

        // CodeMarauder.dev vibes (neon-on-dark, dev/cyber).
        'codemarauder' => [
            'label' => 'CodeMarauder Neon (Lime)',
            'vars'  => [
                '--wf-form-bg'       => 'rgba(5,10,20,.20)',
                '--wf-form-bg2'      => 'rgba(3,6,12,.34)',
                '--wf-form-border'   => 'rgba(163,230,53,.20)',
                '--wf-form-text'     => '#e5e7eb',
                '--wf-label'         => '#f1f5f9',
                '--wf-muted'         => 'rgba(148,163,184,.78)',

                '--wf-field-bg'      => 'rgba(3,6,12,.30)',
                '--wf-field-text'    => '#e5e7eb',
                '--wf-field-border'  => 'rgba(163,230,53,.22)',
                '--wf-placeholder'   => 'rgba(148,163,184,.72)',

                '--wf-accent'        => '#a3e635',

                '--wf-button-bg'     => 'rgba(163,230,53,.16)',
                '--wf-button-text'   => '#f8fafc',
                '--wf-button-border' => 'rgba(163,230,53,.62)',

                '--wf-notice-bg'     => 'rgba(3,6,12,.28)',
                '--wf-notice-border' => 'rgba(163,230,53,.22)',
            ],
        ],
        'synthwave' => [
            'label' => 'Synthwave (Pink/Indigo)',
            'vars'  => [
                '--wf-form-bg'       => 'rgba(15,10,35,.18)',
                '--wf-form-bg2'      => 'rgba(10,5,25,.34)',
                '--wf-form-border'   => 'rgba(244,114,182,.22)',
                '--wf-form-text'     => '#f1f5f9',
                '--wf-label'         => '#f8fafc',
                '--wf-muted'         => 'rgba(203,213,225,.74)',

                '--wf-field-bg'      => 'rgba(10,5,25,.30)',
                '--wf-field-text'    => '#f1f5f9',
                '--wf-field-border'  => 'rgba(129,140,248,.26)',
                '--wf-placeholder'   => 'rgba(203,213,225,.70)',

                '--wf-accent'        => '#f472b6',

                '--wf-button-bg'     => 'rgba(244,114,182,.16)',
                '--wf-button-text'   => '#ffffff',
                '--wf-button-border' => 'rgba(129,140,248,.70)',

                '--wf-notice-bg'     => 'rgba(10,5,25,.28)',
                '--wf-notice-border' => 'rgba(244,114,182,.22)',
            ],
        ],

        // Dark variants (battle / steel / void).
        'tripledark' => [
            'label' => 'TripleDark (Teal)',
            'vars'  => [
                '--wf-form-bg'       => 'rgba(14,17,23,.22)',
                '--wf-form-bg2'      => 'rgba(10,12,16,.34)',
                '--wf-form-border'   => 'rgba(79,209,197,.22)',
                '--wf-form-text'     => '#e5edf7',
                '--wf-label'         => '#e5edf7',
                '--wf-muted'         => 'rgba(156,170,199,.78)',

                '--wf-field-bg'      => 'rgba(14,17,23,.34)',
                '--wf-field-text'    => '#e5edf7',
                '--wf-field-border'  => 'rgba(79,209,197,.24)',
                '--wf-placeholder'   => 'rgba(156,170,199,.72)',

                '--wf-accent'        => '#4fd1c5',

                '--wf-button-bg'     => 'rgba(79,209,197,.14)',
                '--wf-button-text'   => '#e5edf7',
                '--wf-button-border' => 'rgba(79,209,197,.62)',

                '--wf-notice-bg'     => 'rgba(14,17,23,.26)',
                '--wf-notice-border' => 'rgba(79,209,197,.22)',
            ],
        ],
        'stormsteel' => [
            'label' => 'StormSteel (Blue)',
            'vars'  => [
                '--wf-form-bg'       => 'rgba(2,6,23,.16)',
                '--wf-form-bg2'      => 'rgba(2,6,23,.30)',
                '--wf-form-border'   => 'rgba(96,165,250,.20)',
                '--wf-form-text'     => '#e2e8f0',
                '--wf-label'         => '#f1f5f9',
                '--wf-muted'         => 'rgba(148,163,184,.78)',

                '--wf-field-bg'      => 'rgba(2,6,23,.26)',
                '--wf-field-text'    => '#e2e8f0',
                '--wf-field-border'  => 'rgba(96,165,250,.22)',
                '--wf-placeholder'   => 'rgba(148,163,184,.72)',

                '--wf-accent'        => '#60a5fa',

                '--wf-button-bg'     => 'rgba(96,165,250,.16)',
                '--wf-button-text'   => '#f8fafc',
                '--wf-button-border' => 'rgba(96,165,250,.64)',

                '--wf-notice-bg'     => 'rgba(2,6,23,.26)',
                '--wf-notice-border' => 'rgba(96,165,250,.20)',
            ],
        ],
        'voidwalker' => [
            'label' => 'Voidwalker (Purple)',
            'vars'  => [
                '--wf-form-bg'       => 'rgba(8,7,18,.18)',
                '--wf-form-bg2'      => 'rgba(2,2,7,.34)',
                '--wf-form-border'   => 'rgba(167,139,250,.20)',
                '--wf-form-text'     => '#ede9fe',
                '--wf-label'         => '#f5f3ff',
                '--wf-muted'         => 'rgba(216,180,254,.70)',

                '--wf-field-bg'      => 'rgba(2,2,7,.30)',
                '--wf-field-text'    => '#ede9fe',
                '--wf-field-border'  => 'rgba(167,139,250,.24)',
                '--wf-placeholder'   => 'rgba(216,180,254,.68)',

                '--wf-accent'        => '#a78bfa',

                '--wf-button-bg'     => 'rgba(167,139,250,.16)',
                '--wf-button-text'   => '#ffffff',
                '--wf-button-border' => 'rgba(167,139,250,.66)',

                '--wf-notice-bg'     => 'rgba(2,2,7,.28)',
                '--wf-notice-border' => 'rgba(167,139,250,.22)',
            ],
        ],
        'waranima' => [
            'label' => 'Waranima (Violet)',
            'vars'  => [
                '--wf-form-bg'       => 'rgba(2,6,23,.18)',
                '--wf-form-bg2'      => 'rgba(2,6,23,.30)',
                '--wf-form-border'   => 'rgba(139,92,246,.20)',
                '--wf-form-text'     => '#e5e7eb',
                '--wf-label'         => '#f5f3ff',
                '--wf-muted'         => 'rgba(156,163,175,.78)',

                '--wf-field-bg'      => 'rgba(2,6,23,.26)',
                '--wf-field-text'    => '#e5e7eb',
                '--wf-field-border'  => 'rgba(139,92,246,.22)',
                '--wf-placeholder'   => 'rgba(156,163,175,.72)',

                '--wf-accent'        => '#8b5cf6',

                '--wf-button-bg'     => 'rgba(139,92,246,.18)',
                '--wf-button-text'   => '#f5f3ff',
                '--wf-button-border' => 'rgba(167,139,250,.70)',

                '--wf-notice-bg'     => 'rgba(2,6,23,.26)',
                '--wf-notice-border' => 'rgba(139,92,246,.20)',
            ],
        ],

        // Warm themes (ember / sun / inferno).
        'ember' => [
            'label' => 'Ember (Gold)',
            'vars'  => [
                '--wf-form-bg'       => 'rgba(15,10,0,.14)',
                '--wf-form-bg2'      => 'rgba(8,6,2,.30)',
                '--wf-form-border'   => 'rgba(246,173,85,.24)',
                '--wf-form-text'     => '#fff7ed',
                '--wf-label'         => '#ffedd5',
                '--wf-muted'         => 'rgba(253,186,116,.70)',

                '--wf-field-bg'      => 'rgba(8,6,2,.28)',
                '--wf-field-text'    => '#fff7ed',
                '--wf-field-border'  => 'rgba(246,173,85,.26)',
                '--wf-placeholder'   => 'rgba(253,186,116,.68)',

                '--wf-accent'        => '#f6ad55',

                '--wf-button-bg'     => 'rgba(246,173,85,.16)',
                '--wf-button-text'   => '#ffffff',
                '--wf-button-border' => 'rgba(246,173,85,.74)',

                '--wf-notice-bg'     => 'rgba(8,6,2,.26)',
                '--wf-notice-border' => 'rgba(246,173,85,.24)',
            ],
        ],
        'sunforged' => [
            'label' => 'Sunforged (Orange)',
            'vars'  => [
                '--wf-form-bg'       => 'rgba(20,10,4,.16)',
                '--wf-form-bg2'      => 'rgba(10,5,2,.32)',
                '--wf-form-border'   => 'rgba(251,146,60,.24)',
                '--wf-form-text'     => '#fff7ed',
                '--wf-label'         => '#ffedd5',
                '--wf-muted'         => 'rgba(254,215,170,.68)',

                '--wf-field-bg'      => 'rgba(10,5,2,.28)',
                '--wf-field-text'    => '#fff7ed',
                '--wf-field-border'  => 'rgba(251,146,60,.26)',
                '--wf-placeholder'   => 'rgba(254,215,170,.66)',

                '--wf-accent'        => '#fb923c',

                '--wf-button-bg'     => 'rgba(251,146,60,.16)',
                '--wf-button-text'   => '#ffffff',
                '--wf-button-border' => 'rgba(251,146,60,.72)',

                '--wf-notice-bg'     => 'rgba(10,5,2,.26)',
                '--wf-notice-border' => 'rgba(251,146,60,.24)',
            ],
        ],
        'inferno' => [
            'label' => 'Inferno (Red)',
            'vars'  => [
                '--wf-form-bg'       => 'rgba(20,2,6,.16)',
                '--wf-form-bg2'      => 'rgba(10,1,3,.34)',
                '--wf-form-border'   => 'rgba(248,113,113,.22)',
                '--wf-form-text'     => '#fff1f2',
                '--wf-label'         => '#ffe4e6',
                '--wf-muted'         => 'rgba(253,164,175,.70)',

                '--wf-field-bg'      => 'rgba(10,1,3,.28)',
                '--wf-field-text'    => '#fff1f2',
                '--wf-field-border'  => 'rgba(248,113,113,.24)',
                '--wf-placeholder'   => 'rgba(253,164,175,.68)',

                '--wf-accent'        => '#fb7185',

                '--wf-button-bg'     => 'rgba(248,113,113,.16)',
                '--wf-button-text'   => '#ffffff',
                '--wf-button-border' => 'rgba(248,113,113,.68)',

                '--wf-notice-bg'     => 'rgba(10,1,3,.26)',
                '--wf-notice-border' => 'rgba(248,113,113,.22)',
            ],
        ],

        // Nature / rune themes.
        'runeblade' => [
            'label' => 'Runeblade (Emerald)',
            'vars'  => [
                '--wf-form-bg'       => 'rgba(2,10,6,.16)',
                '--wf-form-bg2'      => 'rgba(1,6,3,.34)',
                '--wf-form-border'   => 'rgba(52,211,153,.22)',
                '--wf-form-text'     => '#ecfdf5',
                '--wf-label'         => '#d1fae5',
                '--wf-muted'         => 'rgba(110,231,183,.72)',

                '--wf-field-bg'      => 'rgba(1,6,3,.28)',
                '--wf-field-text'    => '#ecfdf5',
                '--wf-field-border'  => 'rgba(52,211,153,.24)',
                '--wf-placeholder'   => 'rgba(110,231,183,.70)',

                '--wf-accent'        => '#34d399',

                '--wf-button-bg'     => 'rgba(52,211,153,.16)',
                '--wf-button-text'   => '#ffffff',
                '--wf-button-border' => 'rgba(52,211,153,.70)',

                '--wf-notice-bg'     => 'rgba(1,6,3,.26)',
                '--wf-notice-border' => 'rgba(52,211,153,.22)',
            ],
        ],
        'duskwood' => [
            'label' => 'Duskwood (Moss)',
            'vars'  => [
                '--wf-form-bg'       => 'rgba(6,12,6,.16)',
                '--wf-form-bg2'      => 'rgba(3,8,4,.34)',
                '--wf-form-border'   => 'rgba(132,204,22,.20)',
                '--wf-form-text'     => '#f7fee7',
                '--wf-label'         => '#ecfccb',
                '--wf-muted'         => 'rgba(190,242,100,.70)',

                '--wf-field-bg'      => 'rgba(3,8,4,.28)',
                '--wf-field-text'    => '#f7fee7',
                '--wf-field-border'  => 'rgba(132,204,22,.22)',
                '--wf-placeholder'   => 'rgba(190,242,100,.68)',

                '--wf-accent'        => '#84cc16',

                '--wf-button-bg'     => 'rgba(132,204,22,.16)',
                '--wf-button-text'   => '#ffffff',
                '--wf-button-border' => 'rgba(132,204,22,.70)',

                '--wf-notice-bg'     => 'rgba(3,8,4,.26)',
                '--wf-notice-border' => 'rgba(132,204,22,.20)',
            ],
        ],
        'frostforge' => [
            'label' => 'Frostforge (Cyan)',
            'vars'  => [
                '--wf-form-bg'       => 'rgba(2,8,12,.14)',
                '--wf-form-bg2'      => 'rgba(1,4,6,.30)',
                '--wf-form-border'   => 'rgba(34,211,238,.20)',
                '--wf-form-text'     => '#ecfeff',
                '--wf-label'         => '#cffafe',
                '--wf-muted'         => 'rgba(103,232,249,.70)',

                '--wf-field-bg'      => 'rgba(1,4,6,.26)',
                '--wf-field-text'    => '#ecfeff',
                '--wf-field-border'  => 'rgba(34,211,238,.22)',
                '--wf-placeholder'   => 'rgba(103,232,249,.68)',

                '--wf-accent'        => '#22d3ee',

                '--wf-button-bg'     => 'rgba(34,211,238,.14)',
                '--wf-button-text'   => '#ffffff',
                '--wf-button-border' => 'rgba(34,211,238,.66)',

                '--wf-notice-bg'     => 'rgba(1,4,6,.24)',
                '--wf-notice-border' => 'rgba(34,211,238,.20)',
            ],
        ],

        // Light theme (for bright sites).
        'parchment' => [
            'label' => 'Parchment (Light)',
            'vars'  => [
                '--wf-form-bg'       => '#ffffff',
                '--wf-form-bg2'      => '#f8fafc',
                '--wf-form-border'   => 'rgba(15,23,42,.18)',
                '--wf-form-text'     => '#0f172a',
                '--wf-label'         => '#0f172a',
                '--wf-muted'         => 'rgba(51,65,85,.75)',

                '--wf-field-bg'      => '#ffffff',
                '--wf-field-text'    => '#0f172a',
                '--wf-field-border'  => 'rgba(15,23,42,.22)',
                '--wf-placeholder'   => 'rgba(71,85,105,.65)',

                '--wf-accent'        => '#2563eb',

                '--wf-button-bg'     => 'rgba(37,99,235,.10)',
                '--wf-button-text'   => '#0f172a',
                '--wf-button-border' => 'rgba(37,99,235,.42)',

                '--wf-notice-bg'     => '#f0f6fc',
                '--wf-notice-border' => 'rgba(37,99,235,.35)',
            ],
        ],
    ];

    // Allow other plugins/themes to add more presets.
    return apply_filters('warstyles_presets', $presets);
}


function warstyles_get_form_style(int $form_id): array {
    $enabled = get_post_meta($form_id, '_warstyles_enabled', true);
    $enabled = ($enabled === '' || $enabled === '1' || $enabled === 1); // default ON

    $preset = (string) get_post_meta($form_id, '_warstyles_preset', true);
    $preset = $preset !== '' ? sanitize_key($preset) : 'warchief';

    $accent = (string) get_post_meta($form_id, '_warstyles_accent', true);
    $accent = $accent !== '' ? sanitize_hex_color($accent) : '';

    $stretch = (string) get_post_meta($form_id, '_warstyles_stretch', true);
    $stretch = ($stretch === '1' || $stretch === 1);

    $bg_mode = (string) get_post_meta($form_id, '_warstyles_bg_mode', true);
    $bg_mode = in_array($bg_mode, ['solid','gradient'], true) ? $bg_mode : 'solid';

    // Color overrides (all optional)
    $form_bg      = (string) get_post_meta($form_id, '_warstyles_form_bg', true);
    $form_bg      = $form_bg !== '' ? sanitize_hex_color($form_bg) : '';

    $form_bg2     = (string) get_post_meta($form_id, '_warstyles_form_bg2', true);
    $form_bg2     = $form_bg2 !== '' ? sanitize_hex_color($form_bg2) : '';

    $form_text    = (string) get_post_meta($form_id, '_warstyles_form_text', true);
    $form_text    = $form_text !== '' ? sanitize_hex_color($form_text) : '';

    $label        = (string) get_post_meta($form_id, '_warstyles_label', true);
    $label        = $label !== '' ? sanitize_hex_color($label) : '';

    $field_bg     = (string) get_post_meta($form_id, '_warstyles_field_bg', true);
    $field_bg     = $field_bg !== '' ? sanitize_hex_color($field_bg) : '';

    $field_text   = (string) get_post_meta($form_id, '_warstyles_field_text', true);
    $field_text   = $field_text !== '' ? sanitize_hex_color($field_text) : '';

    $field_border = (string) get_post_meta($form_id, '_warstyles_field_border', true);
    $field_border = $field_border !== '' ? sanitize_hex_color($field_border) : '';

    $placeholder  = (string) get_post_meta($form_id, '_warstyles_placeholder', true);
    $placeholder  = $placeholder !== '' ? sanitize_hex_color($placeholder) : '';

    $button_bg    = (string) get_post_meta($form_id, '_warstyles_button_bg', true);
    $button_bg    = $button_bg !== '' ? sanitize_hex_color($button_bg) : '';

    $button_text  = (string) get_post_meta($form_id, '_warstyles_button_text', true);
    $button_text  = $button_text !== '' ? sanitize_hex_color($button_text) : '';

    $button_border = (string) get_post_meta($form_id, '_warstyles_button_border', true);
    $button_border = $button_border !== '' ? sanitize_hex_color($button_border) : '';

    $form_border   = (string) get_post_meta($form_id, '_warstyles_form_border', true);
    $form_border   = $form_border !== '' ? sanitize_hex_color($form_border) : '';

    return [
        'enabled'       => $enabled,
        'preset'        => $preset,
        'accent'        => $accent,
        'stretch'       => $stretch,
        'bg_mode'       => $bg_mode,

        'form_bg'       => $form_bg,
        'form_bg2'      => $form_bg2,
        'form_text'     => $form_text,
        'label'         => $label,
        'field_bg'      => $field_bg,
        'field_text'    => $field_text,
        'field_border'  => $field_border,
        'placeholder'   => $placeholder,
        'button_bg'     => $button_bg,
        'button_text'   => $button_text,
        'button_border' => $button_border,
        'form_border'   => $form_border,
    ];
}

function warstyles_enqueue_frontend_assets(): void {
    static $done = false;
    if ($done) return;
    $done = true;

    wp_register_style('warstyles-frontend', WARSTYLES_URL . 'assets/frontend.css', [], WARSTYLES_VERSION);
    wp_enqueue_style('warstyles-frontend');
}

function warstyles_inline_style_for_form(int $form_id, array $style): void {
    static $seen = [];
    if (isset($seen[$form_id])) return;
    $seen[$form_id] = true;

    $presets = warstyles_get_presets();
    $preset = $style['preset'] ?? 'warchief';
    if (!isset($presets[$preset])) $preset = 'warchief';

    $vars = $presets[$preset]['vars'] ?? [];
    if (!is_array($vars)) $vars = [];

    // Apply per-form overrides (all optional).
    if (!empty($style['accent']))        $vars['--wf-accent'] = $style['accent'];
    if (!empty($style['form_text']))     $vars['--wf-form-text'] = $style['form_text'];
    if (!empty($style['label']))         $vars['--wf-label'] = $style['label'];
    if (!empty($style['field_bg']))      $vars['--wf-field-bg'] = $style['field_bg'];
    if (!empty($style['field_text']))    $vars['--wf-field-text'] = $style['field_text'];
    if (!empty($style['field_border']))  $vars['--wf-field-border'] = $style['field_border'];
    if (!empty($style['placeholder']))   $vars['--wf-placeholder'] = $style['placeholder'];
    if (!empty($style['button_bg']))     $vars['--wf-button-bg'] = $style['button_bg'];
    if (!empty($style['button_text']))   $vars['--wf-button-text'] = $style['button_text'];
    if (!empty($style['button_border'])) $vars['--wf-button-border'] = $style['button_border'];
    if (!empty($style['form_border']))   $vars['--wf-form-border'] = $style['form_border'];

    // Background: solid or gradient (both hex colors).
    $bg_mode = isset($style['bg_mode']) ? sanitize_key((string)$style['bg_mode']) : 'solid';
    $form_bg = $style['form_bg'] ?? '';
    $form_bg2 = $style['form_bg2'] ?? '';

    if ($form_bg) {
        $vars['--wf-form-bg'] = $form_bg;
    }
    if ($form_bg2) {
        $vars['--wf-form-bg2'] = $form_bg2;
    }

    $sel = '.warforms-reboot-form.warstyles-form-' . (int)$form_id;

    $css = $sel . '{';
    foreach ($vars as $k => $v) {
        $k = (string)$k;
        $v = (string)$v;
        if ($k === '' || $v === '') continue;
        if (strpos($k, '--') !== 0) continue;
        $css .= $k . ':' . $v . ';';
    }
    // background mode token (no raw CSS)
    $css .= '--wf-bg-mode:' . ($bg_mode === 'gradient' ? 'gradient' : 'solid') . ';';
    $css .= '}';

    wp_add_inline_style('warstyles-frontend', $css);
}

/**
 * Inject classes into the [warform] output without touching WarForms.
 * We do this via do_shortcode_tag so it survives core updates.
 */
add_filter('do_shortcode_tag', function($output, $tag, $attr) {
    if ($tag !== 'warform') return $output;
    if (!is_string($output) || $output === '') return $output;
    if (!warstyles_is_warforms_present()) return $output;

    $form_id = isset($attr['id']) ? (int)$attr['id'] : 0;
    if ($form_id <= 0) return $output;

    $style = warstyles_get_form_style($form_id);
    if (empty($style['enabled'])) return $output;

    // Enqueue CSS and add per-form variables.
    add_action('wp_enqueue_scripts', 'warstyles_enqueue_frontend_assets', 1);
    warstyles_enqueue_frontend_assets();
    warstyles_inline_style_for_form($form_id, $style);

    $preset = sanitize_key((string)($style['preset'] ?? 'warchief'));

    $classes = [
        'warstyles',
        'warstyles-preset-' . $preset,
        'warstyles-form-' . $form_id,
    ];
    if (!empty($style['stretch'])) {
        $classes[] = 'warstyles-stretch';
    }
    if (!empty($style['bg_mode']) && $style['bg_mode'] === 'gradient') {
        $classes[] = 'warstyles-bg-gradient';
    }
    $add_classes = ' ' . implode(' ', array_map('sanitize_html_class', $classes));

    // Fast-path: replace the known class attribute.
    $needle = 'class="warforms-reboot-form"';
    if (strpos($output, $needle) !== false) {
        return str_replace($needle, 'class="warforms-reboot-form' . esc_attr($add_classes) . '"', $output);
    }

    // Fallback: regex add to the <form ... class="...">.
    $out = preg_replace(
        '/<form\b([^>]*)\bclass="([^"]*)"/i',
        '<form$1class="$2' . esc_attr($add_classes) . '"',
        $output,
        1
    );

    return is_string($out) && $out !== '' ? $out : $output;
}, 10, 3);

/**
 * Admin UI: meta box on warform edit screen.
 */
add_action('add_meta_boxes', function() {
    if (!warstyles_is_warforms_present()) return;

    // If WarForms supports the integrated tabbed builder UI, WarStyles lives inside that UI (not in the sidebar).
    if (function_exists('warforms_reboot_admin_builder_supports_tabs') && warforms_reboot_admin_builder_supports_tabs()) {
        return;
    }

    add_meta_box(
        'warstyles_metabox',
        'WarStyles',
        'warstyles_render_metabox',
        'warform',
        'side',
        'default'
    );
});

function warstyles_render_metabox($post): void {
    if (!current_user_can('edit_post', $post->ID)) return;

    wp_nonce_field('warstyles_save_' . $post->ID, 'warstyles_nonce');

    $style   = warstyles_get_form_style((int)$post->ID);
    $presets = warstyles_get_presets();

    $enabled = !empty($style['enabled']);
    $preset  = isset($style['preset']) ? sanitize_key((string)$style['preset']) : 'warchief';
    if (!isset($presets[$preset])) $preset = 'warchief';

    $bg_mode = isset($style['bg_mode']) && $style['bg_mode'] === 'gradient' ? 'gradient' : 'solid';

    // Compute preview CSS variables (preset + overrides) so preview renders even if JS is blocked.
    $vars = isset($presets[$preset]['vars']) && is_array($presets[$preset]['vars']) ? $presets[$preset]['vars'] : [];
    if (!is_array($vars)) $vars = [];

    if (!empty($style['accent']))        $vars['--wf-accent'] = $style['accent'];
    if (!empty($style['form_text']))     $vars['--wf-form-text'] = $style['form_text'];
    if (!empty($style['label']))         $vars['--wf-label'] = $style['label'];
    if (!empty($style['field_bg']))      $vars['--wf-field-bg'] = $style['field_bg'];
    if (!empty($style['field_text']))    $vars['--wf-field-text'] = $style['field_text'];
    if (!empty($style['field_border']))  $vars['--wf-field-border'] = $style['field_border'];
    if (!empty($style['placeholder']))   $vars['--wf-placeholder'] = $style['placeholder'];
    if (!empty($style['button_bg']))     $vars['--wf-button-bg'] = $style['button_bg'];
    if (!empty($style['button_text']))   $vars['--wf-button-text'] = $style['button_text'];
    if (!empty($style['button_border'])) $vars['--wf-button-border'] = $style['button_border'];
    if (!empty($style['form_border']))   $vars['--wf-form-border'] = $style['form_border'];
    if (!empty($style['form_bg']))       $vars['--wf-form-bg'] = $style['form_bg'];
    if (!empty($style['form_bg2']))      $vars['--wf-form-bg2'] = $style['form_bg2'];

    $preview_style = '';
    foreach ($vars as $k => $v) {
        $k = (string)$k;
        $v = (string)$v;
        if ($k === '' || $v === '') continue;
        if (strpos($k, '--') !== 0) continue;
        $preview_style .= $k . ':' . $v . ';';
    }

    echo '<div class="wf-panel warstyles-wf-panel">';
    echo '<div class="warstyles-builder-grid warstyles-view-controls" data-warstyles-form-id="' . (int)$post->ID . '">';

    echo '<h2 class="nav-tab-wrapper warstyles-subtabs">';
    echo '  <a href="#" class="nav-tab nav-tab-active" data-warstyles-view="controls">Controls</a>';
    echo '  <a href="#" class="nav-tab" data-warstyles-view="preview">Preview</a>';
    echo '  <a href="#" class="nav-tab" data-warstyles-view="css">CSS Editor</a>';
    echo '</h2>';

    echo '<div class="warstyles-subviews">';
    echo '<div class="warstyles-subview warstyles-subview-controls">';

    // Sidebar
    echo '<div class="warstyles-sidebar">';

    echo '<div class="warstyles-head">';
    echo '  <label class="warstyles-enable"><input type="checkbox" name="warstyles_enabled" value="1" ' . checked($enabled, true, false) . '> <strong>Enable WarStyles</strong></label>';
    echo '  <p class="description" style="margin:6px 0 0">Color-only theming (no layout changes on the front-end).</p>';
    echo '</div>';

    echo '<div class="warstyles-section">';
    echo '  <h4>Preset</h4>';
    echo '  <select id="warstyles_preset" name="warstyles_preset" style="width:100%">';
    foreach ($presets as $key => $p) {
        $key = sanitize_key((string)$key);
        $label = isset($p['label']) ? (string)$p['label'] : $key;
        printf('<option value="%s" %s>%s</option>', esc_attr($key), selected($preset, $key, false), esc_html($label));
    }
    echo '  </select>';
    echo '  <div class="warstyles-row warstyles-row-tight" style="margin-top:8px">';
    echo '    <button type="button" class="button warstyles-reset-overrides">Reset overrides</button>';
    echo '    <span class="warstyles-status" aria-live="polite"></span>';
    echo '  </div>';
    echo '  <p class="description" style="margin:8px 0 0">Changing presets clears overrides so the preset fully applies.</p>';
    echo '</div>';

    echo '<div class="warstyles-section">';
    echo '  <h4>Layout</h4>';
    echo '  <label class="warstyles-check"><input type="checkbox" id="warstyles_stretch" name="warstyles_stretch" value="1" ' . checked(!empty($style['stretch']), true, false) . '> Stretch form to container width</label>';

    echo '  <label style="margin-top:10px; display:block;"><strong>Background mode</strong></label>';
    echo '  <select name="warstyles_bg_mode" id="warstyles_bg_mode" style="width:100%">';
    printf('<option value="solid" %s>Solid</option>', selected($bg_mode, 'solid', false));
    printf('<option value="gradient" %s>Gradient (two colors)</option>', selected($bg_mode, 'gradient', false));
    echo '  </select>';
    echo '</div>';

    echo '<details class="warstyles-details" open>';
    echo '  <summary><strong>Accent + Text</strong></summary>';
    echo '  <div class="warstyles-details-body">';
    echo '    <label for="warstyles_accent"><strong>Accent</strong></label>';
    echo '    <input type="text" id="warstyles_accent" class="warstyles-color" name="warstyles_accent" value="' . esc_attr($style['accent'] ?? '') . '" placeholder="#38bdf8" style="width:100%">';
    echo '    <label for="warstyles_form_text" style="margin-top:10px"><strong>Form text</strong></label>';
    echo '    <input type="text" id="warstyles_form_text" class="warstyles-color" name="warstyles_form_text" value="' . esc_attr($style['form_text'] ?? '') . '" placeholder="#e5edf7" style="width:100%">';
    echo '    <label for="warstyles_label" style="margin-top:10px"><strong>Label text</strong></label>';
    echo '    <input type="text" id="warstyles_label" class="warstyles-color" name="warstyles_label" value="' . esc_attr($style['label'] ?? '') . '" placeholder="#dbe6ff" style="width:100%">';
    echo '  </div>';
    echo '</details>';

    echo '<details class="warstyles-details">';
    echo '  <summary><strong>Fields</strong></summary>';
    echo '  <div class="warstyles-details-body">';
    echo '    <label for="warstyles_field_bg"><strong>Field background</strong></label>';
    echo '    <input type="text" id="warstyles_field_bg" class="warstyles-color" name="warstyles_field_bg" value="' . esc_attr($style['field_bg'] ?? '') . '" placeholder="#0b1222" style="width:100%">';
    echo '    <label for="warstyles_field_text" style="margin-top:10px"><strong>Field text</strong></label>';
    echo '    <input type="text" id="warstyles_field_text" class="warstyles-color" name="warstyles_field_text" value="' . esc_attr($style['field_text'] ?? '') . '" placeholder="#e5edf7" style="width:100%">';
    echo '    <label for="warstyles_field_border" style="margin-top:10px"><strong>Field border</strong></label>';
    echo '    <input type="text" id="warstyles_field_border" class="warstyles-color" name="warstyles_field_border" value="' . esc_attr($style['field_border'] ?? '') . '" placeholder="#1f2937" style="width:100%">';
    echo '    <label for="warstyles_placeholder" style="margin-top:10px"><strong>Placeholder</strong></label>';
    echo '    <input type="text" id="warstyles_placeholder" class="warstyles-color" name="warstyles_placeholder" value="' . esc_attr($style['placeholder'] ?? '') . '" placeholder="#9caac7" style="width:100%">';
    echo '  </div>';
    echo '</details>';

    echo '<details class="warstyles-details">';
    echo '  <summary><strong>Form surface</strong></summary>';
    echo '  <div class="warstyles-details-body">';
    echo '    <label for="warstyles_form_bg"><strong>BG color 1</strong></label>';
    echo '    <input type="text" id="warstyles_form_bg" class="warstyles-color" name="warstyles_form_bg" value="' . esc_attr($style['form_bg'] ?? '') . '" placeholder="#05070a" style="width:100%">';
    echo '    <label for="warstyles_form_bg2" style="margin-top:10px"><strong>BG color 2</strong></label>';
    echo '    <input type="text" id="warstyles_form_bg2" class="warstyles-color" name="warstyles_form_bg2" value="' . esc_attr($style['form_bg2'] ?? '') . '" placeholder="#0b1222" style="width:100%">';
    echo '    <label for="warstyles_form_border" style="margin-top:10px"><strong>Form border</strong></label>';
    echo '    <input type="text" id="warstyles_form_border" class="warstyles-color" name="warstyles_form_border" value="' . esc_attr($style['form_border'] ?? '') . '" placeholder="#1f2937" style="width:100%">';
    echo '    <p class="description" style="margin-top:8px">Tip: leave a color blank to use the preset default.</p>';
    echo '  </div>';
    echo '</details>';

    echo '<details class="warstyles-details">';
    echo '  <summary><strong>Button</strong></summary>';
    echo '  <div class="warstyles-details-body">';
    echo '    <label for="warstyles_button_bg"><strong>Button BG</strong></label>';
    echo '    <input type="text" id="warstyles_button_bg" class="warstyles-color" name="warstyles_button_bg" value="' . esc_attr($style['button_bg'] ?? '') . '" placeholder="#38bdf8" style="width:100%">';
    echo '    <label for="warstyles_button_text" style="margin-top:10px"><strong>Button text</strong></label>';
    echo '    <input type="text" id="warstyles_button_text" class="warstyles-color" name="warstyles_button_text" value="' . esc_attr($style['button_text'] ?? '') . '" placeholder="#ffffff" style="width:100%">';
    echo '    <label for="warstyles_button_border" style="margin-top:10px"><strong>Button border</strong></label>';
    echo '    <input type="text" id="warstyles_button_border" class="warstyles-color" name="warstyles_button_border" value="' . esc_attr($style['button_border'] ?? '') . '" placeholder="#38bdf8" style="width:100%">';
    echo '  </div>';
    echo '</details>';

    echo '</div>'; // sidebar
    echo '</div>'; // subview-controls

    echo '<div class="warstyles-subview warstyles-subview-preview">';


    // Preview
    $preview_classes = [
        'warforms-reboot-form',
        'warstyles',
        'warstyles-preview-form',
        'warstyles-preset-' . $preset,
    ];
    if (!empty($style['stretch'])) $preview_classes[] = 'warstyles-stretch';
    if ($bg_mode === 'gradient') $preview_classes[] = 'warstyles-bg-gradient';

    echo '<div class="warstyles-main">';
    echo '  <div class="warstyles-preview-head">';
    echo '    <h3 style="margin:0">Live preview</h3>';
    echo '    <p class="description" style="margin:6px 0 0">Mocked-up form rendering inside this editor. Preview updates as you change colors.</p>';
    echo '  </div>';

    echo '  <div class="warstyles-preview-shell">';
    echo '    <div class="warstyles-preview-page">';
    echo '      <aside class="warstyles-preview-sidebar" aria-hidden="true">';
    echo '        <div class="warstyles-preview-brand">Warchief / CodeMarauder</div>';
    echo '        <div class="warstyles-preview-nav">';
    echo '          <div class="warstyles-preview-nav-item is-active">Forms</div>';
    echo '          <div class="warstyles-preview-nav-item">Docs</div>';
    echo '          <div class="warstyles-preview-nav-item">Contact</div>';
    echo '        </div>';
    echo '      </aside>';

    echo '      <main class="warstyles-preview-content">';
    echo '        <div class="warstyles-preview-card">';
    echo '          <div id="warstyles-preview-form" class="' . esc_attr(implode(' ', $preview_classes)) . '" style="' . esc_attr($preview_style) . '">';
    echo '            <div class="wf-notice wf-success">Preview only — styles update live.</div>';

    // A simple mocked layout with a 2-col row + additional fields.
    echo '            <div class="wf-row wf-cols-2">';
    echo '              <div class="wf-col">';
    echo '                <div class="wf-field wf-type-text">';
    echo '                  <label>Full Name <span aria-hidden="true">*</span></label>';
    echo '                  <input type="text" value="Aria Ray" placeholder="e.g. Aria Ray" >';
    echo '                </div>';
    echo '              </div>';
    echo '              <div class="wf-col">';
    echo '                <div class="wf-field wf-type-email">';
    echo '                  <label>Email <span aria-hidden="true">*</span></label>';
    echo '                  <input type="email" value="aria@warchief.dev" placeholder="you@example.com" >';
    echo '                </div>';
    echo '              </div>';
    echo '            </div>';

    echo '            <div class="wf-field wf-type-url">';
    echo '              <label>Website</label>';
    echo '              <input type="url" value="https://codemarauder.dev" placeholder="https://..." >';
    echo '              <div class="wf-description">Optional: share your homepage.</div>';
    echo '            </div>';

    echo '            <div class="wf-row wf-cols-2">';
    echo '              <div class="wf-col">';
    echo '                <div class="wf-field wf-type-select">';
    echo '                  <label>Faction</label>';
    echo '                  <select >';
    echo '                    <option>— Select —</option>';
    echo '                    <option selected>Vanguard</option>';
    echo '                    <option>Sentinel</option>';
    echo '                    <option>Marauder</option>';
    echo '                  </select>';
    echo '                </div>';
    echo '              </div>';
    echo '              <div class="wf-col">';
    echo '                <div class="wf-field wf-type-textarea">';
    echo '                  <label>Message</label>';
    echo '                  <textarea rows="4" placeholder="Write a quick note..." >Looking to style a WarForm with WarStyles presets.</textarea>';
    echo '                  <div class="wf-error">Sample error state (for preview).</div>';
    echo '                </div>';
    echo '              </div>';
    echo '            </div>';

    echo '            <div class="wf-field wf-type-checkbox_group">';
    echo '              <label>Updates</label>';
    echo '              <div class="wf-choice-group">';
    echo '                <label><input type="checkbox" checked > Patch notes</label>';
    echo '                <label><input type="checkbox" > Release alerts</label>';
    echo '                <label><input type="checkbox" > Beta invites</label>';
    echo '              </div>';
    echo '            </div>';

    echo '            <p style="margin-top:12px"><button type="submit" >Submit</button></p>';

    echo '          </div>'; // #warstyles-preview-form
    echo '        </div>'; // card
    echo '      </main>';
    echo '    </div>'; // page
    echo '  </div>'; // shell

    echo '</div>'; // main
    echo '</div>'; // subview-preview

    $custom_css = get_post_meta($post->ID, '_warstyles_custom_css', true);
    if (!is_string($custom_css)) $custom_css = '';

    echo '<div class="warstyles-subview warstyles-subview-css">';
    echo '  <div class="warstyles-css-editor">';
    echo '    <h3 style="margin:0 0 8px">CSS Style Editor</h3>';
    echo '    <p class="description" style="margin:0 0 10px">Add custom CSS for this WarForm. (Stored now; applied on the front-end in a later phase.)</p>';
    echo '    <textarea name="warstyles_custom_css" id="warstyles_custom_css" rows="18" style="width:100%;font-family:ui-monospace,SFMono-Regular,Menlo,Monaco,Consolas,\"Liberation Mono\",\"Courier New\",monospace">' . esc_textarea($custom_css) . '</textarea>';
    echo '  </div>';
    echo '</div>'; // subview-css

    echo '</div>'; // subviews

    echo '</div>'; // grid
    echo '</div>'; // wf-panel
}

add_action('admin_enqueue_scripts', function($hook) {
    if (!warstyles_is_warforms_present()) return;

    // Only on warform edit screens.
    $screen = function_exists('get_current_screen') ? get_current_screen() : null;
    if (!$screen || $screen->post_type !== 'warform') return;

    wp_enqueue_style('wp-color-picker');

    // Preview in the builder tab uses the frontend CSS so it looks like the real form.
    if (defined('WARFORMS_REBOOT_URL') && defined('WARFORMS_REBOOT_VERSION')) {
        wp_enqueue_style('warforms-reboot-frontend', WARFORMS_REBOOT_URL . 'assets/frontend.css', [], WARFORMS_REBOOT_VERSION);
    }
    warstyles_enqueue_frontend_assets();

    wp_enqueue_script('warstyles-admin', WARSTYLES_URL . 'assets/admin.js', ['wp-color-picker', 'jquery'], WARSTYLES_VERSION, true);
    wp_enqueue_style('warstyles-admin', WARSTYLES_URL . 'assets/admin.css', [], WARSTYLES_VERSION);

    // Provide presets for live preview + preset apply behavior.
    $presets = warstyles_get_presets();
    $payload = [];
    foreach ($presets as $key => $p) {
        $key = sanitize_key((string)$key);
        if ($key === '') continue;

        $payload[$key] = [
            'label' => isset($p['label']) ? (string)$p['label'] : $key,
            'vars'  => (isset($p['vars']) && is_array($p['vars'])) ? $p['vars'] : [],
        ];
    }

    wp_localize_script('warstyles-admin', 'WARSTYLES_ADMIN', [
        'presets' => $payload,
    ]);
});

add_action('save_post_warform', function($post_id) {
    if (!warstyles_is_warforms_present()) return;
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (!isset($_POST['warstyles_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['warstyles_nonce'])), 'warstyles_save_' . $post_id)) return;
    if (!current_user_can('edit_post', $post_id)) return;

    $enabled = isset($_POST['warstyles_enabled']) ? '1' : '0';
    update_post_meta($post_id, '_warstyles_enabled', $enabled);

    $preset = isset($_POST['warstyles_preset']) ? sanitize_key(wp_unslash($_POST['warstyles_preset'])) : '';
    $presets = warstyles_get_presets();
    if ($preset === '' || !isset($presets[$preset])) $preset = 'warchief';
    update_post_meta($post_id, '_warstyles_preset', $preset);

    $stretch = isset($_POST['warstyles_stretch']) ? '1' : '0';
    update_post_meta($post_id, '_warstyles_stretch', $stretch);

    $bg_mode = isset($_POST['warstyles_bg_mode']) ? sanitize_key(wp_unslash($_POST['warstyles_bg_mode'])) : 'solid';
    if (!in_array($bg_mode, ['solid','gradient'], true)) $bg_mode = 'solid';
    update_post_meta($post_id, '_warstyles_bg_mode', $bg_mode);

    $hex_fields = [
        '_warstyles_accent'        => 'warstyles_accent',
        '_warstyles_form_text'     => 'warstyles_form_text',
        '_warstyles_label'         => 'warstyles_label',
        '_warstyles_field_bg'      => 'warstyles_field_bg',
        '_warstyles_field_text'    => 'warstyles_field_text',
        '_warstyles_field_border'  => 'warstyles_field_border',
        '_warstyles_placeholder'   => 'warstyles_placeholder',
        '_warstyles_form_bg'       => 'warstyles_form_bg',
        '_warstyles_form_bg2'      => 'warstyles_form_bg2',
        '_warstyles_form_border'   => 'warstyles_form_border',
        '_warstyles_button_bg'     => 'warstyles_button_bg',
        '_warstyles_button_text'   => 'warstyles_button_text',
        '_warstyles_button_border' => 'warstyles_button_border',
    ];

    foreach ($hex_fields as $meta_key => $post_key) {
        $val = isset($_POST[$post_key]) ? sanitize_hex_color(wp_unslash($_POST[$post_key])) : '';
        if ($val) {
            update_post_meta($post_id, $meta_key, $val);
        } else {
            delete_post_meta($post_id, $meta_key);
        }
    }

    // Custom CSS (stored now; applied on the front-end in a later phase).
    $custom_css = isset($_POST['warstyles_custom_css']) ? wp_unslash($_POST['warstyles_custom_css']) : '';
    if (!is_string($custom_css)) $custom_css = '';
    $custom_css = trim($custom_css);
    if ($custom_css !== '') {
        update_post_meta($post_id, '_warstyles_custom_css', $custom_css);
    } else {
        delete_post_meta($post_id, '_warstyles_custom_css');
    }

});
